% *************************************************************************
% 
%                      iGem Modelling - 2018
%                     Delft University of Technology
% 
% *************************************************************************
% 
%               Title
% 
%                    File: Steady State Analysis of Red Blood Cell
%                    Production
% 
%              
% *************************************************************************
clear all
close all

tic
%Create range of EPO steady states
EPO_t = linspace(0,75,10000);

% CFU-E Apoptosis Rate variables
a1 = 0.35; b1 = 0.07;
c1 = 3; k1 = 0.14;

%Calculate CFU-E Apoptosis Rate
alpha_q = (a1-b1)./(1+exp(k1*EPO_t-c1)) + b1;

%Bone Marrow Reticulocyte Maturation Velocity variables
a2 = 2; b2 = 0.35;
c2 = 2.3; k2 = 0.2;

%Calculate Reticulocyte Maturation Velocity
vs = (a2-b2)./(1+exp(-k2*EPO_t+c2)) + b2;
Transit_Time =  3 - (3-0.75)*(vs-0.5)/(2-0.5); 

%Initial BFU-E population at maturity of 0 days
S0 = 10^8;

% Proliferation Rates
Bp = 0.2; %/day
Bq = 0.57; %/day
Br = 1.024; %day

% Apoptosis Rates
alpha_s = 0.089; %/day
alpha_m = 0.005; %/day

%Neocytolysis Parameters
tau_E = 9.8;
c_E = 3.5;
k_E = 3;
b_E = 0.1;

%Start and End points
BFU_E_start = 0; %BFU-E start
CFU_E_start = 7; %CFU-E start
Erythroblasts_start = 13; %Erythroblasts start
Reticulocyte_start = 18; %Reticulocyte start
Erythrocytes_start1 = 21; %Erythrocytes start1
Erythrocytes_start2 = 35; %Erythrocytes start2
Erythrocytes_start3 = 42; %Erythrocytes start3
Erythrocytes_end = 141; %Erythrocytes end3


%Calculate BFU-E cells Entering CFU-E stage
BFU_E_end = S0*exp(Bp*CFU_E_start);

%Caluculate BFU-E count
BFU_E_fun =  @(mu) S0*exp(Bp .* mu);
BFU_E_count = integral(BFU_E_fun,0,CFU_E_start);

%Initiallize SS Red Blood Cell range vector
mesh = 1000;

%Intialize SS Red Blood Cell count
Progenitor_count = [];
Erythroblasts_count = [];
RBC_count_total = [];

for ii = 1:length(EPO_t)
    
    %CFU-E function
    CFU_E_end = BFU_E_end*exp((Bq - alpha_q(ii))*(Erythroblasts_start - CFU_E_start));
    CFU_E_fun = @(mu) BFU_E_end*exp((Bq - alpha_q(ii)).* mu);
    CFU_E_count = integral(CFU_E_fun,0,Erythroblasts_start - CFU_E_start);
    Progenitor_count = [Progenitor_count (BFU_E_count + CFU_E_count)];
    
    %Erythroblasts function
    Erythroblasts_end = CFU_E_end*exp(Br*(Reticulocyte_start - Erythroblasts_start));
    Erythroblasts_fun = @(mu) CFU_E_end*exp(Br .* mu);
    Erythroblasts_total = integral(Erythroblasts_fun, 0, Reticulocyte_start - Erythroblasts_start);
    Erythroblasts_count = [Erythroblasts_count Erythroblasts_total];
    
    %Reticulocytes fuction
    Reticulocyte_end = Erythroblasts_end*exp(-alpha_s*(Transit_Time(ii)));
    
    %Red Blood Cell functions
    RBC_fun1 = @(mu) Reticulocyte_end*exp(-alpha_m .* mu);
    RBC_end1 = RBC_fun1(Erythrocytes_start2-(Transit_Time(ii)+Reticulocyte_start));
    RBC_count1 = integral(RBC_fun1,0,Erythrocytes_start2-(Transit_Time(ii)+Reticulocyte_start));
    
    if EPO_t(ii) < tau_E
            alpha_m_n = alpha_m + min(c_E/EPO_t(ii)^k_E , b_E);
        else
            alpha_m_n = alpha_m;
    end
    
    RBC_fun2 = @(mu) RBC_end1*exp(-alpha_m_n .* mu);
    RBC_end2 = RBC_fun2(Erythrocytes_start3-Erythrocytes_start2);
    RBC_count2 = integral(RBC_fun2,0,Erythrocytes_start3-Erythrocytes_start2);
    
    RBC_fun3 = @(mu) RBC_end2*exp(-alpha_m .* mu);
    RBC_end3 = RBC_fun3(Erythrocytes_end- Erythrocytes_start3);
    RBC_count3 = integral(RBC_fun3,0,Erythrocytes_end- Erythrocytes_start3);
    
    RBC_count = RBC_count1 + RBC_count2 + RBC_count3;
    
    RBC_count_total = [RBC_count_total RBC_count];
end

%Figure
figure(1)
plot(EPO_t, RBC_count_total/1e12, 'r', 'LineWidth',2); hold on;
plot([9.8 9.8], [0 120],'--b' )
plot([3.271066310 3.271066310], [0 120], 'k--')
plot([0 75], [40 40], '--r', 'LineWidth',2)
xlabel('EPO Concentration (mU/mL)', 'FontSize', 14) 
ylabel('Red Blood Cell Population (# of cells x10^{12})', 'FontSize', 14)
legend('Steady State Red blood cell count', 'Neocytolysis Trigger (EPO < 9.8 mU/ml)', ...
    'Maximum Neocytolysis Rate Reached (EPO < 3.3 mU/ml)', 'Stroke Risk (RBC Population > 40x10^{12})')
set(gcf,'color','w'); 
set(gca,'fontsize',16);
set(gcf, 'Position', [500, 200, 700, 600])
xlim([0 75])


Steady_State_RBC = interp1(EPO_t,RBC_count_total/1e12,9.8)
Steady_State_strokeEPO = interp1(RBC_count_total/1e12,EPO_t, 40)
toc